unit SuperReg;

// Copyright  2002 by Ziff Davis Media, Inc.
// Written by Neil J. Rubenking

// "Super" TRegistry class. The existing TRegistry will throw an
// exception if you try to read an Integer and the value is actually
// stored as a string, or try to write an Integer or String when
// the value is already present with a different type. TSuperReg
// can safely read and write String or Integer values, converting
// as necessary. It can also force-write a string or integer,
// by simply deleting any existing value first.

interface
USES Windows, SysUtils, Registry;

type
  TSuperReg = class(TRegistry)
  public
    function ReadIntegerSafe(const Name: string; default : Integer)
      : Integer;
    function ReadStringSafe(const Name, default: string): string;
    procedure WriteIntegerSafe(const Name: string; Value: Integer);
    procedure WriteStringSafe(const Name, Value: string);
    procedure WriteIntegerForce(const Name: string; Value: Integer);
    procedure WriteStringForce(const Name, Value: string);
  end;

implementation

{ TSuperReg }

function TSuperReg.ReadIntegerSafe(const Name: string;
  default: Integer): Integer;
VAR
  rd  : TRegDataType;
  siz : Integer;
begin
  Result := Default;
  IF NOT ValueExists(Name) THEN Exit;
  Siz := GetData(Name, nil, 0, rd);
  CASE rd OF
    // String? Convert to Integer
    rdString,
    rdExpandString : Result := StrToIntDef(ReadString(Name), Default);
    // Integer? Just use it
    rdInteger      : Result := ReadInteger(Name);
    // Other? If it's 4 bytes in size, copy the data
    rdUnknown,
    rdBinary       : IF Siz = 4 THEN GetData(Name, @Result, 4, rd);
  END;
end;

function TSuperReg.ReadStringSafe(const Name, default: string): string;
VAR
  rd : TRegDataType;
  siz : Integer;
begin
  Result := Default;
  IF NOT ValueExists(Name) THEN Exit;
  Siz := GetData(Name, nil, 0, rd);
  CASE rd OF
    // String? Just read it
    rdString,
    rdExpandString : Result := ReadString(Name);
    // Integer? Convert to string
    rdInteger      : Result := IntToStr(ReadInteger(Name));
    // Other? Treat it as a string
    rdUnknown,
    rdBinary : begin
      SetLength(Result, Siz);
      GetData(Name, PChar(Result), Siz, rd);
      Result := StrPas(PChar(Result));
    end;
  END;
end;

procedure TSuperReg.WriteIntegerSafe(const Name: string;
  Value: Integer);
VAR
  DType : DWord;
  DSize : DWord;
begin
  IF RegQueryValueEx(CurrentKey, PChar(Name), nil, @DType, nil,
    @DSize) <> ERROR_SUCCESS THEN
    WriteInteger(Name, Value)
  ELSE
    begin
      CASE DType OF
        REG_SZ         : WriteString(Name, IntToStr(Value));
        REG_EXPAND_SZ  : WriteExpandString(Name, IntToStr(Value));
        REG_DWORD_BIG_ENDIAN :
                         WriteInteger(Name, MakeLong(Swap(HiWord(Value)),
                           Swap(LoWOrd(Value))));
        REG_DWORD      : WriteInteger(Name, Value);
        ELSE
          RegSetValueEx(CurrentKey, PChar(Name), 0, DType,
            @Value, 4);
      END;
    end;
end;

procedure TSuperReg.WriteStringSafe(const Name, Value: string);
VAR
  DType : DWord;
  DSize : DWord;
  I     : Integer;
begin
  IF RegQueryValueEx(CurrentKey, PChar(Name), nil, @DType, nil,
    @DSize) <> ERROR_SUCCESS THEN
    WriteString(Name, Value)
  ELSE
    begin
      I := StrToIntDef(Value, 0);
      CASE DType OF
        REG_SZ         : WriteString(Name, Value);
        REG_EXPAND_SZ  : WriteExpandString(Name, Value);
        REG_DWORD_BIG_ENDIAN :
                         WriteInteger(Name, MakeLong(Swap(HiWord(I)),
                           Swap(LoWOrd(I))));
        REG_DWORD      : WriteInteger(Name, I);
        ELSE
          RegSetValueEx(CurrentKey, PChar(Name), 0, DType,
            PChar(Value), Length(Value)+1);
      END;
    end;
end;

procedure TSuperReg.WriteIntegerForce(const Name: string;
  Value: Integer);
begin
  DeleteValue(Name);
  WriteInteger(Name, Value);
end;

procedure TSuperReg.WriteStringForce(const Name, Value: string);
begin
  DeleteValue(Name);
  WriteString(Name, Value);
end;

end.
